#!/usr/bin/env bash
#
# b64_encode.sh – Base‑64 encode a binary file (single‑line output)
#
# Usage:
#   ./b64_encode.sh <binary_input> <b64_output>
#
#   <binary_input>  – path to the file you want to encode (e.g., ct.bin)
#   <b64_output>    – path where the Base‑64 string will be written
#
# The script produces a **single line** of Base‑64 characters (no
# line‑breaks).  It aborts with a clear error message if the input
# file does not exist or cannot be read.

set -euo pipefail

# ----------------------------------------------------------------------
# Helper: print usage and exit
# ----------------------------------------------------------------------
usage() {
    echo "Usage: $0 <binary_input> <b64_output>"
    exit 1
}
[[ $# -eq 2 ]] || usage

BIN_IN="$1"
B64_OUT="$2"

# ----------------------------------------------------------------------
# Basic sanity checks
# ----------------------------------------------------------------------
if [[ ! -f "$BIN_IN" ]]; then
    echo "ERROR: Input file '$BIN_IN' does not exist or is not a regular file." >&2
    exit 1
fi

if [[ ! -r "$BIN_IN" ]]; then
    echo "ERROR: Cannot read input file '$BIN_IN' (permission denied)." >&2
    exit 1
fi

# ----------------------------------------------------------------------
# Perform the encoding
# ----------------------------------------------------------------------
# The `-w 0` flag (GNU coreutils) disables line wrapping.
# macOS/BSD `base64` does not have `-w`; it already emits a single line,
# so we detect the platform and adjust accordingly.
if base64 --help 2>&1 | grep -q '\-w'; then
    # GNU coreutils (Linux)
    base64 -w 0 "$BIN_IN" > "$B64_OUT"
else
    # BSD/macOS – already single‑line output
    base64 "$BIN_IN" > "$B64_OUT"
fi

# ----------------------------------------------------------------------
# Summary
# ----------------------------------------------------------------------
echo "[INFO] Base‑64 encoded $(stat -c%s "$BIN_IN") bytes → $B64_OUT"
exit 0